from flask import Blueprint, request, current_app, jsonify, send_from_directory,send_file
import json
from flask_jwt_extended import jwt_required, get_jwt_identity
from datetime import datetime
from . import db
from .AuthContext import token_required
from .models import Article, ArticleImage, User, Comment, Category
from .utils import allowed_file, save_image, fetch_jubatimes_articles
import os

main = Blueprint('main', __name__)

@main.route('/')
def index():
    return jsonify({'message': 'Juba Times API', 'version': '1.0.0'})

@main.route('/api/getimage/<uploads>/<image>/<int:img_id>/<filename>')
def serve_image(uploads, image, img_id, filename):
    uploads_path = os.path.join(current_app.instance_path, 'uploads', 'images', str(img_id))
    return send_from_directory(uploads_path, filename)

@main.route('/uploads/<path:path>')
def serve_upload(path):
    return send_from_directory(os.path.join('instance', 'uploads'), path)


@main.route('/api/topstories/<section>')
def get_jbtimes_stories(section):
    data = fetch_jubatimes_articles(section)
    if data:
        return jsonify(data)
    return jsonify({'error': 'Failed to fetch data from JubaTimes'}), 500

@main.route('/api/stories')
def jbtimes_stories():
    try:
        all_articles = Article.query.filter_by(is_published=True).all()
        
        if not all_articles:
            return jsonify({
                'status': 'success',
                'data': {
                    'featured': [],
                    'trending': [],
                    'top_stories': [],
                    'recent': []
                },
                'metadata': {
                    'total': 0,
                    'timestamp': datetime.utcnow().isoformat()
                }
            }), 200
        
        featured = [article.to_jbtimes_format() for article in all_articles if article.is_featured][:5]
        
        trending = sorted([article.to_jbtimes_format() for article in all_articles], 
                         key=lambda x: x.get('view_count', 0), reverse=True)[:5]
        
        top_stories = sorted([article.to_jbtimes_format() for article in all_articles], 
                            key=lambda x: x.get('published_date', ''), reverse=True)[:10]
        
        recent = sorted([article.to_jbtimes_format() for article in all_articles], 
                       key=lambda x: x.get('published_date', ''), reverse=True)[:20]
        
        return jsonify({
            'status': 'success',
            'data': {
                'featured': featured,
                'trending': trending,
                'top_stories': top_stories,
                'recent': recent
            },
            'metadata': {
                'total': len(all_articles),
                'timestamp': datetime.utcnow().isoformat(),
                'categories': list(set([article.section for article in all_articles]))
            }
        }), 200
        
    except Exception as e:
        return jsonify({
            'status': 'error',
            'error': str(e),
            'message': 'Failed to fetch stories'
        }), 500  

@main.route('/api/story/<section>')
def jbtimes_section():
    data = Article.query.all()
    if data:
        return jsonify([article.to_jbtimes_format() for article in data])
    return jsonify({'error': 'Failed to fetch data from JubaTimes'}), 500  

@main.route('/api/storysearch/<story>')
def jbtimes_story(story):
    data = Article.query.msearch(story,limit=10).all()
    if data:
        return jsonify([article.to_jbtimes_format() for article in data])
    return jsonify({'error': 'Failed to fetch data from JubaTimes'}), 500  


@main.route('/api/storysection/<search>', methods=["POST", "GET"])	
def search(search):
    try:
        all_articles = Article.query.msearch(search,limit=10).all()
        if not all_articles:
            return jsonify({
                'status': 'success',
                'data': {
                    'featured': [],
                    'trending': [],
                    'top_stories': [],
                    'recent': []
                },
                'metadata': {
                    'total': 0,
                    'timestamp': datetime.utcnow().isoformat()
                }
            }), 200
        
        featured = [article.to_jbtimes_format() for article in all_articles if article.is_featured][:5]
        
        trending = sorted([article.to_jbtimes_format() for article in all_articles], 
                         key=lambda x: x.get('view_count', 0), reverse=True)[:5]
        
        top_stories = sorted([article.to_jbtimes_format() for article in all_articles], 
                            key=lambda x: x.get('published_date', ''), reverse=True)[:10]
        
        recent = sorted([article.to_jbtimes_format() for article in all_articles], 
                       key=lambda x: x.get('published_date', ''), reverse=True)[:20]
        
        return jsonify({
            'status': 'success',
            'data': {
                'featured': featured,
                'trending': trending,
                'top_stories': top_stories,
                'recent': recent
            },
            'metadata': {
                'total': len(all_articles),
                'timestamp': datetime.utcnow().isoformat(),
                'categories': list(set([article.section for article in all_articles]))
            }
        }), 200
        
    except Exception as e:
        return jsonify({
            'status': 'error',
            'error': str(e),
            'message': 'Failed to fetch stories'
        }), 500  


@main.route('/articles', methods=['POST'])
@token_required
def post_article(current_user):
    try:
        current_user_id = current_user.id
        user = User.query.get(current_user_id)
        if not user:
            return jsonify({'error': 'User not found'}), 404
        
        if not user.is_active:
            return jsonify({'error': 'Account deactivated'}), 403
        
        data = request.form

        if not data or not data.get('title') or not data.get('content'):
            return jsonify({'error': 'Title and content are required'}), 400

        import uuid
        article_uri = f"juba://article/{uuid.uuid4().hex}"
 
        print(data.get('category_id'))
        article = Article(
            uri=article_uri,
            url=data.get('url', f"http://localhost:5000/article/{uuid.uuid4().hex}"),
            short_url=data.get('url', f"https://jubatimes.com/article"),
            title=data['title'],
            abstract=data.get('abstract'),
            byline=data.get('byline', f"By {current_user.username}"),
            content= data.get('content'),
            section=data.get('section', 'general'),
            subsection=data.get('subsection'),
            author_id=current_user_id,
            category_id=3, 
            item_type=data.get('item_type', 'Article'),
            material_type_facet='facet',
            kicker='kicker',
            nytimes_id=2,
            des_facet=data.get('des_facet', []),
            geo_facet=data.get('geo_facet', []),
            per_facet=data.get('per_facet', []),
            org_facet=data.get('org_facet', []),
            is_published=True,
            is_featured=data.get('is_featured', 'False').lower() == 'true'
        )
        
        db.session.add(article)
        db.session.flush()  

        if 'image' in request.files:
            image_file = request.files['image']
            if image_file and image_file.filename != '':
                image_caption = request.form.get('image_caption', '')
                image_url, thumbnail_url = save_image(image_file, article.id)
                
                article_image = ArticleImage(
                    article_id=article.id,
                    image_url=image_url,
                    thumbnail_url=thumbnail_url,
                    caption=image_caption,
                    is_primary=True
                )
                db.session.add(article_image)
        
        db.session.commit()
        
        return jsonify({
            'message': 'Article created successfully',
            'article': article.to_jbtimes_format()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        print(f"Error creating article: {str(e)}") 
        return jsonify({'error': str(e)}), 500


@main.route('/articles/<article_id>', methods=['GET'])
def get_articles(article_id):
    try:
        article = Article.query.get_or_404(article_id)
        
        if not article.is_published:
            return jsonify({'error': 'Article not found'}), 404
        
        article.view_count += 1
        db.session.commit()
        #print(article.to_dict())
        return jsonify({'article': article.to_dict()}), 200
        
    except Exception as e:
        print(e)
        return jsonify({'error': str(e)}), 500

@main.route('/article/<article_id>', methods=['GET'])
def get_article(article_id):
    try:
        article = Article.query.get_or_404(article_id)
        
        if not article.is_published:
            return jsonify({'error': 'Article not found'}), 404
        
        article.view_count += 1
        db.session.commit()
        return jsonify({'article': article.to_dict()}), 200
        
    except Exception as e:
        print(e)
        return jsonify({'error': str(e)}), 500
    
@main.route('/get-image')
def get_image():
    image_path = 'static/my_image.png' 
    return send_file(image_path, mimetype='image/png')


@main.route('/articles/<int:article_id>/images', methods=['POST'])
@jwt_required()
def upload_article_image(article_id):
    try:
        user_id = get_jwt_identity()
        article = Article.query.get_or_404(article_id)
        
   
        user = User.query.get(user_id)
        if article.author_id != user_id and not user.is_admin:
            return jsonify({'error': 'Unauthorized'}), 403
        
        if 'image' not in request.files:
            return jsonify({'error': 'No image provided'}), 400
        
        file = request.files['image']
        if file.filename == '':
            return jsonify({'error': 'No selected file'}), 400
        
        image_url, thumbnail_url = save_image(file, article_id)
        
        if not image_url:
            return jsonify({'error': 'Invalid file type'}), 400
        
        image = ArticleImage(
            article_id=article_id,
            image_url=image_url,
            thumbnail_url=thumbnail_url,
            caption=request.form.get('caption'),
            is_primary=request.form.get('is_primary', 'false').lower() == 'true'
        )
        
        db.session.add(image)
        db.session.commit()
        
        return jsonify({
            'message': 'Image uploaded successfully',
            'image': image.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'error': str(e)}), 500


@main.route('/articles/<int:article_id>/comments', methods=['GET'])
def get_comments(article_id):
    try:
        comments = Comment.query.filter_by(
            article_id=article_id, 
            is_approved=True
        ).order_by(Comment.created_at.desc()).all()
        
        return jsonify({'comments': [comment.to_dict() for comment in comments]}), 200
        
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@main.route('/articles/<int:article_id>/comments', methods=['POST'])
@jwt_required()
def create_comment(article_id):
    try:
        user_id = get_jwt_identity()
        data = request.get_json()
        
        if not data or not data.get('content'):
            return jsonify({'error': 'Comment content is required'}), 400
        
        comment = Comment(
            article_id=article_id,
            user_id=user_id,
            content=data['content'],
            is_approved=False  
        )
        
        db.session.add(comment)
        db.session.commit()
        
        return jsonify({
            'message': 'Comment submitted for moderation',
            'comment': comment.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'error': str(e)}), 500


@main.route('/categories', methods=['GET'])
def get_categories():
    try:
        categories = Category.query.all()
        return jsonify({'categories': [cat.to_dict() for cat in categories]}), 200
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@main.route('/categories/<int:category_id>/articles', methods=['GET'])
def get_articles_by_category(category_id):
    try:
        page = request.args.get('page', 1, type=int)
        per_page = request.args.get('per_page', 10, type=int)
        
        category = Category.query.get_or_404(category_id)
        articles = Article.query.filter_by(
            category_id=category_id, 
            is_published=True
        ).order_by(Article.published_date.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return jsonify({
            'category': category.name,
            'articles': [article.to_jbtimes_format() for article in articles.items],
            'total': articles.total,
            'pages': articles.pages,
            'current_page': page
        }), 200
        
    except Exception as e:
        return jsonify({'error': str(e)}), 500
    
