from . import db, bcrypt
from datetime import datetime
from sqlalchemy.dialects.sqlite import JSON
from  werkzeug.security import generate_password_hash, check_password_hash

class User(db.Model):
    __tablename__ = 'users'
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(80), unique=True, nullable=False)
    email = db.Column(db.String(120), unique=True, nullable=False)
    password_hash = db.Column(db.String(128), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    is_active = db.Column(db.Boolean, default=True)
    is_admin = db.Column(db.Boolean, default=False)
    
    articles = db.relationship('Article', backref='author', lazy=True)
    comments = db.relationship('Comment', backref='user', lazy=True)
    
    def set_password(self, password):
        self.password_hash=generate_password_hash(password)
    
    def check_password(self, password):
        return check_password_hash(self.password_hash, password)
    
    def to_dict(self):
        return {
            'id': self.id,
            'username': self.username,
            'email': self.email,
            'created_at': self.created_at.isoformat(),
            'is_admin': self.is_admin
        }

class Category(db.Model):
    __tablename__ = 'categories'
    
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(50), unique=True, nullable=False)
    slug = db.Column(db.String(50), unique=True, nullable=False)
    description = db.Column(db.Text, nullable=True)

    articles = db.relationship('Article', backref='category_rel', lazy=True)
    
    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'slug': self.slug,
            'description': self.description,
            'article_count': len(self.articles)
        }

class Article(db.Model):
    __tablename__ = 'articles'
    __searchable__ = ['title','abstract', 'content','section'] 
    id = db.Column(db.Integer, primary_key=True)
    uri = db.Column(db.String(100), unique=True, nullable=False)
    url = db.Column(db.String(500), nullable=False)
    short_url = db.Column(db.String(100), nullable=True)
    title = db.Column(db.String(200), nullable=False)
    abstract = db.Column(db.Text, nullable=True)
    byline = db.Column(db.String(200), nullable=True)
    content = db.Column(db.Text)
    section = db.Column(db.String(50), nullable=False)
    subsection = db.Column(db.String(50), nullable=True)
    author_id = db.Column(db.Integer, db.ForeignKey('users.id'), nullable=False)
    category_id = db.Column(db.Integer, db.ForeignKey('categories.id'), nullable=True)
    published_date = db.Column(db.DateTime, default=datetime.utcnow)
    updated_date = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    item_type = db.Column(db.String(50), default='Article')
    material_type_facet = db.Column(db.String(100), nullable=True)
    kicker = db.Column(db.String(200), nullable=True)
    is_published = db.Column(db.Boolean, default=False)
    is_featured = db.Column(db.Boolean, default=False)
    view_count = db.Column(db.Integer, default=0)
    nytimes_id = db.Column(db.String(50), nullable=True)
    
    des_facet = db.Column(JSON, default=[])
    geo_facet = db.Column(JSON, default=[])
    per_facet = db.Column(JSON, default=[])
    org_facet = db.Column(JSON, default=[])
    
    images = db.relationship('ArticleImage', backref='article', lazy=True)
    comments = db.relationship('Comment', backref='article', lazy=True)
    def to_dict(self):
        image_url = self.images[0].image_url if self.images else None
        return {
            'id': self.id,
            'url': self.url,
            'title': self.title,
            'abstract': self.abstract,
            'content': self.content,
            'section': self.section,
            'subsection': self.subsection,
            'author_id': self.author_id,
            'category_id': self.category_id,
            'published_date': self.published_date.isoformat() if self.published_date else None,
            'updated_date': self.updated_date.isoformat() if self.updated_date else None,
            'item_type': self.item_type,
            'material_type_facet': self.material_type_facet,
            'kicker': self.kicker,
            'is_published': self.is_published,
            'is_featured': self.is_featured,
            'view_count': self.view_count,
            'jbtimes_id': self.nytimes_id,
            'des_facet': self.des_facet,
            'geo_facet': self.geo_facet,
            'per_facet': self.per_facet,
            'org_facet': self.org_facet,
            'image_url': image_url
           
        }
    
    def to_jbtimes_format(self):
        multimedia = []
        if self.images:
            for img in self.images:
                multimedia.append({
                    'url': img.image_url,
                    'format': 'Standard Thumbnail',
                    'height': 75,
                    'width': 75,
                    'type': 'image',
                    'subtype': 'photo',
                    'caption': img.caption,
                    'copyright': 'The Juba Times'
                })
                
        
        return {
            'id': self.id,
            'uri': self.uri,
            'url': self.url,
            'short_url': self.short_url,
            'title': self.title,
            'abstract': self.abstract,
            'byline': self.byline,
            'content': self.content,
            'created_date': self.created_date.isoformat() if hasattr(self, 'created_date') else self.published_date.isoformat(),
            'published_date': self.published_date.isoformat(),
            'updated_date': self.updated_date.isoformat(),
            'item_type': self.item_type,
            'material_type_facet': self.material_type_facet or "",
            'kicker': self.kicker or "",
            'section': self.section,
            'subsection': self.subsection or "",
            'jbtimes_id': self.nytimes_id,
            'is_featured': self.is_featured,
            'des_facet': self.des_facet or [],
            'geo_facet': self.geo_facet or [],
            'per_facet': self.per_facet or [],
            'org_facet': self.org_facet or [],
            'multimedia': multimedia,
            'category': self.category_rel.name if self.category_rel else self.section
        }

class ArticleImage(db.Model):
    __tablename__ = 'article_images'
    
    id = db.Column(db.Integer, primary_key=True)
    article_id = db.Column(db.Integer, db.ForeignKey('articles.id'), nullable=False)
    image_url = db.Column(db.String(500), nullable=False)
    thumbnail_url = db.Column(db.String(500), nullable=True)
    caption = db.Column(db.String(200), nullable=True)
    copyright = db.Column(db.String(100), nullable=True)
    format = db.Column(db.String(50), nullable=True)
    height = db.Column(db.Integer, nullable=True)
    width = db.Column(db.Integer, nullable=True)
    type = db.Column(db.String(50), default='image')
    subtype = db.Column(db.String(50), default='photo')
    is_primary = db.Column(db.Boolean, default=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    
    def to_dict(self):
        return {
            'url': self.image_url,
            'format': self.format or 'Standard Thumbnail',
            'height': self.height,
            'width': self.width,
            'type': self.type,
            'subtype': self.subtype,
            'caption': self.caption,
            'copyright': self.copyright
        }

class Comment(db.Model):
    __tablename__ = 'comments'
    
    id = db.Column(db.Integer, primary_key=True)
    article_id = db.Column(db.Integer, db.ForeignKey('articles.id'), nullable=False)
    user_id = db.Column(db.Integer, db.ForeignKey('users.id'), nullable=False)
    content = db.Column(db.Text, nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    is_approved = db.Column(db.Boolean, default=False)
    
    def to_dict(self):
        return {
            'id': self.id,
            'article_id': self.article_id,
            'user': self.user.username,
            'content': self.content,
            'created_at': self.created_at.isoformat(),
            'is_approved': self.is_approved
        }